// Fill Suno's custom creation form with lyrics from MusicMaker.
// Runs as a Chrome extension content script on suno.com/create.

(async function fillSunoForm() {
  const { suno_transfer } = await chrome.storage.local.get('suno_transfer');
  if (!suno_transfer) return;

  // Ignore stale data (> 5 min old)
  if (Date.now() - suno_transfer.ts > 5 * 60 * 1000) {
    chrome.storage.local.remove('suno_transfer');
    return;
  }

  const { title, style, lyrics } = suno_transfer;

  // Wait for the page to render
  await waitFor(() => document.querySelectorAll('button').length > 3, 10000);

  // Switch to Custom mode
  for (const btn of document.querySelectorAll('button')) {
    if (btn.textContent.trim() === 'Custom') {
      btn.click();
      break;
    }
  }

  // Wait for Custom form textareas to appear
  await waitFor(
    () => [...document.querySelectorAll('textarea')].some(t => t.maxLength === 1000),
    5000
  );

  // Small delay for React to settle
  await sleep(300);

  // Fill textareas: styles has maxlength="1000", lyrics doesn't
  for (const ta of document.querySelectorAll('textarea')) {
    if (ta.maxLength === 1000) {
      setReactValue(ta, style);
    } else {
      setReactValue(ta, lyrics);
    }
  }

  // Fill title input if present
  if (title) {
    for (const input of document.querySelectorAll('input[type="text"], input:not([type])')) {
      const ph = (input.placeholder || '').toLowerCase();
      if (ph.includes('title') || ph.includes('name')) {
        setReactInputValue(input, title);
        break;
      }
    }
  }

  chrome.storage.local.remove('suno_transfer');
})();

function setReactValue(el, value) {
  const setter = Object.getOwnPropertyDescriptor(HTMLTextAreaElement.prototype, 'value').set;
  setter.call(el, value);
  el.dispatchEvent(new Event('input', { bubbles: true }));
  el.dispatchEvent(new Event('change', { bubbles: true }));
}

function setReactInputValue(el, value) {
  const setter = Object.getOwnPropertyDescriptor(HTMLInputElement.prototype, 'value').set;
  setter.call(el, value);
  el.dispatchEvent(new Event('input', { bubbles: true }));
  el.dispatchEvent(new Event('change', { bubbles: true }));
}

function waitFor(predicate, timeout) {
  return new Promise((resolve, reject) => {
    if (predicate()) return resolve();
    const interval = setInterval(() => {
      if (predicate()) { clearInterval(interval); resolve(); }
    }, 200);
    setTimeout(() => { clearInterval(interval); reject(new Error('timeout')); }, timeout);
  });
}

function sleep(ms) {
  return new Promise(r => setTimeout(r, ms));
}
